//------------------------------------------------------------------------------
// File: svr_statGui.cs
// Handles server-side data-sending to client for the town stat page.
// Author: Michael Felice
//------------------------------------------------------------------------------

// for every stat that needs data, a message is received (messages
// are only sent back for stats that need to be displayed)
function serverCmdFindHappyModData(%client, %index)
{
   switch (%index)
   {
   case $HMOD_POPULATION: SendHappyPopulationMod(%client);
   case $HMOD_HUNGER: SendHappyHungerMod(%client);
   case $HMOD_UNEMPLOYED: SendHappyUnemployedMod(%client);
   case $HMOD_HOMELESS: SendHappyHomelessMod(%client);
   case $HMOD_DRUNK: SendHappyDrunkMod(%client);
   case $HMOD_TOUGHSLINGER: SendHappyToughSlingerMod(%client);
   case $HMOD_NICESLINGER: SendHappyNiceSlingerMod(%client);
   case $HMOD_TOUGHSHERIFF: SendHappyToughSheriffMod(%client);
   case $HMOD_NICESHERIFF: SendHappyNiceSheriffMod(%client);
   case $HMOD_BUILDINGBLDG: SendHappyBuildingMod(%client);
   case $HMOD_DAMAGEBLDG: SendHappyDamageMod(%client);
   case $HMOD_PRODUCE: SendHappyProduceMod(%client);
   case $HMOD_BANDITATTACK: SendHappyBanditMod(%client);
   case $HMOD_ANIMALATTACK: SendHappyAnimalMod(%client);
   case $HMOD_TAXES: SendHappyTaxesMod(%client);
   case $HMOD_PARADE: SendHappyParadeMod(%client);
   }
}

// when this message is received, the completiong message needs to
// be returned to the client (in order to display the town stat page)
function serverCmdFindHappyModDone(%client)
{
   commandToClient(%client, 'SetHappyModDone');
}

// unhappiness due to a high population
function SendHappyPopulationMod(%client)
{
   // find the happiness mod based on current town population
   %count = rsGetResourceCount($Resource::People);
   %data = GameResourceData.getTotalPopulationHappiness(%count);
   
   // send the town overpopulation data
   if (%data != 0)
   {
      // find out the level of overpopulation
      %text = "";
      if (%data > -20) %text = "Low";
      else if (%data > -40) %text = "Medium";
      else if (%data > -60) %text = "High";
      else if (%data > -70) %text = "Very High";
      else %text = "Extremely High";
      
      // send the data to the client for display
      commandToClient(%client, 'SetHappyModData', $HMOD_POPULATION, %text @ " " @ %data);
   }
}

// unhappiness due to hungry civilians
function SendHappyHungerMod(%client)
{
   // get the character list for hunger counting
   %list = slgGetServerCharacterList();
   %count = %list.getSize();
   
   // total the hunger
   %hunger = 0;
   for (%index = 0; %index < %count; %index++)
   {
      // make sure the object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // add the character's hunger to the total hunger
      %component = slgQueryInterface(%object, $CID_CHARACTER);
      %hunger += %component.getHunger();
   }
   
   // if the town is hungry, send the hunger happiness mod
   if (%hunger != 0)
   {
      commandToClient(%client, 'SetHappyModData', $HMOD_HUNGER, -%hunger);
   }
}

// unhappiness due to unemployed civilians
function SendHappyUnemployedMod(%client)
{
   // get the character list for unemployment counting
   %list = slgGetServerCharacterList();
   %count = %list.getSize();
   
   // total the unemployment
   %unemployed = 0;
   for (%index = 0; %index < %count; %index++)
   {
      // make sure the object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // add the object's unemployment to the total unemployment
      %unemployed += %object.joblessUnhappy;
   }
   
   // if the town is unemployed, send the unemployed happiness mod
   if (%unemployed != 0)
   {
      commandToClient(%client, 'SetHappyModData', $HMOD_UNEMPLOYED, -%unemployed);
   }
}

// unhappiness due to homeless civilians
function SendHappyHomelessMod(%client)
{
   // get the character list for homeless counting
   %list = slgGetServerCharacterList();
   %count = %list.getSize();
   
   // total the homeless
   %homeless = 0;
   for (%index = 0; %index < %count; %index++)
   {
      // make sure the object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // add the object's homeless to the total homelessness
      %homeless += %object.homelessUnhappy;
   }
   
   // if the town is homeless, send the homeless happiness mod
   if (%homeless != 0)
   {
      commandToClient(%client, 'SetHappyModData', $HMOD_HOMELESS, -%homeless);
   }
}

// unhappiness due to drunk civilians
function SendHappyDrunkMod(%client)
{
   // get the character list for drunks counting
   %list = slgGetServerCharacterList();
   %count = %list.getSize();
   
   // total the drunks
   %drunks = 0;
   for (%index = 0; %index < %count; %index++)
   {
      // make sure the object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // add the object's drunken unhappiness to the total drunks
      %drunks += %object.drunkUnhappy;
   }
   
   // if the town is drunk, send the drunks happiness mod
   if (%drunks != 0)
   {
      commandToClient(%client, 'SetHappyModData', $HMOD_DRUNK, -%drunks);
   }
}

// count the number of tough gunslingers there are
function SendHappyToughSlingerMod(%client)
{
   // get the character list for gunslinger counting
   %list = slgGetServerCharacterList();
   %count = %list.getSize();
   
   // total the unhappiness
   %data = 0;
   for (%index = 0; %index < %count; %index++)
   {
      // make sure the object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // make sure the object is a combat unit
      %component = slgQueryInterface(%object, $CID_COMBAT);
      if (isObject(%component) == false)
      {
         continue;
      }
      
      // make sure the object is not in law enforcement
      %component = slgQueryInterface(%object, $CID_CHARACTER);
      %datablock = %component.getDataBlock();
      if (%datablock.canReformDrunk == true)
      {
         continue;
      }
      
      // check if the object is a tough gunslinger
      if (%datablock.initHappiness < 0)
      {
         %data += %datablock.initHappiness;
      }
   }
   
   // if any tough gunslingers are counted, send the data
   if (%data != 0)
   {
      commandToClient(%client, 'SetHappyModData', $HMOD_TOUGHSLINGER, %data);
   }
}

// count the number of nice gunslingers there are
function SendHappyNiceSlingerMod(%client)
{
   // get the character list for gunslinger counting
   %list = slgGetServerCharacterList();
   %count = %list.getSize();
   
   // total the unhappiness
   %data = 0;
   for (%index = 0; %index < %count; %index++)
   {
      // make sure the object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // make sure the object is a combat unit
      %component = slgQueryInterface(%object, $CID_COMBAT);
      if (isObject(%component) == false)
      {
         continue;
      }
      
      // make sure the object is not in law enforcement
      %component = slgQueryInterface(%object, $CID_CHARACTER);
      %datablock = %component.getDataBlock();
      if (%datablock.canReformDrunk == true)
      {
         continue;
      }
      
      // check if the object is a nice gunslinger
      if (%datablock.initHappiness > 0)
      {
         %data += %datablock.initHappiness;
      }
   }
   
   // if any nice gunslingers are counted, send the data
   if (%data != 0)
   {
      commandToClient(%client, 'SetHappyModData', $HMOD_NICESLINGER, %data);
   }
}

// look for a tough sheriff
function SendHappyToughSheriffMod(%client)
{
   // get the character list for sheriff counting
   %list = slgGetServerCharacterList();
   %count = %list.getSize();
   
   // find a tough sheriff
   for (%index = 0; %index < %count; %index++)
   {
      // make sure the object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // make sure the object is a combat unit
      %component = slgQueryInterface(%object, $CID_COMBAT);
      if (isObject(%component) == false)
      {
         continue;
      }
      
      // make sure the object is in law enforcement
      %component = slgQueryInterface(%object, $CID_CHARACTER);
      %datablock = %component.getDataBlock();
      if (%datablock.canReformDrunk == false)
      {
         continue;
      }
      
      // check if the object is a tough sheriff
      if (%datablock.initHappiness < 0)
      {
         commandToClient(%client, 'SetHappyModData', $HMOD_TOUGHSHERIFF, %datablock.initHappiness);
         return;
      }
   }
}

// look for a nice sheriff
function SendHappyNiceSheriffMod(%client)
{
   // get the character list for sheriff counting
   %list = slgGetServerCharacterList();
   %count = %list.getSize();
   
   // find a nice sheriff
   for (%index = 0; %index < %count; %index++)
   {
      // make sure the object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // make sure the object is a combat unit
      %component = slgQueryInterface(%object, $CID_COMBAT);
      if (isObject(%component) == false)
      {
         continue;
      }
      
      // make sure the object is in law enforcement
      %component = slgQueryInterface(%object, $CID_CHARACTER);
      %datablock = %component.getDataBlock();
      if (%datablock.canReformDrunk == false)
      {
         continue;
      }
      
      // check if the object is a nice sheriff
      if (%datablock.initHappiness > 0)
      {
         commandToClient(%client, 'SetHappyModData', $HMOD_NICESHERIFF, %datablock.initHappiness);
         return;
      }
   }
}

// count the total happiness of buildings of a particular type
function SendHappyBuildingMod(%client)
{
   // get the building list for building happiness counts
   %list = slgGetServerBuildingList();
   %count = %list.getSize();
   
   // build a building count list
   %buildingCount = 0;
   %buildingList = "";
   for (%index = 0; %index < %count; %index++)
   {
      // make sure the object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // make sure the building is in the production state
      %component = slgQueryInterface(%object, $CID_BUILDING);
      if (%component.getState() != $BuildingState::Production)
      {
         continue;
      }
      
      // only add buildings to the list if they have building happiness
      %datablock = %component.getDataBlock();
      if (%datablock.hasBuildHappiness() == false)
      {
         continue;
      }
      
      // add the building type to the count list
      %found = false;
      %type = %datablock.getName();
      for (%buildingIndex = 0; %buildingIndex < %buildingCount; %buildingIndex++)
      {
         %buildingType = getWord(%buildingList[%buildingIndex], 0);
         if (%type $= %buildingType)
         {
            %found = true;
            %number = getWord(%buildingList[%buildingIndex], 1);
            %buildingList[%buildingIndex] = %type @ " " @ %number + 1;
            break;
         }
      }
      
      // if the building was not already in the list, add it to the list
      if (%found == false)
      {
         %buildingList[%buildingCount] = %type @ " 1";
         %buildingCount++;
      }
   }
   
   // now go through the building cound list to determine which of
   // these buildings have stat modifiers that need to be send to
   // the client
   for (%index = 0; %index < %buildingCount; %index++)
   {
      // get the details about the building counts
      %data = %buildingList[%index];
      %type = getWord(%data, 0);
      %count = getWord(%data, 1);
      %happiness = %type.getMaxBuildHappiness(%count - 1);
      
      // send the happiness info if happiness is non-zero from the building
      if (%happiness != 0)
      {
         commandToClient(%client, 'SetHappyModData', $HMOD_BUILDINGBLDG,
            %type.happinessString @ " " @ %happiness);
      }
   }
}

// tallies up building damage and sends it to the client
function SendHappyDamageMod(%client)
{
   // get the building list for building happiness counts
   %list = slgGetServerBuildingList();
   %count = %list.getSize();
   
   // count the happiness lost due to damage
   %damage = 0;
   for (%index = 0; %index < %count; %index++)
   {
      // make sure the object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // make sure the building is in the production state
      %component = slgQueryInterface(%object, $CID_BUILDING);
      if (%component.getState() != $BuildingState::Production)
      {
         continue;
      }
      
      // update the total damage happiness
      %datablock = %component.getDataBlock();
      %percentage = 100 * %object.health / %object.getMaxHealth();
      %damage += %datablock.getDamageHappiness(%percentage);
   }
   
   // if a damage count is found, send it to the client
   if (%damage != 0)
   {
      commandToClient(%client, 'SetHappyModData', $HMOD_DAMAGEBLDG, %damage);
   }
}

// send the varied production states for the buildings
function SendHappyProduceMod(%client)
{
   // determine the happiness boost based on production happiness
   %count = 0;
   for (%index = 0; %index < $LastProduceCount; %index++)
   {
      %count += GameResourceData.getProduceHappiness(%index);
   }
   
   // if there is happiness from production, update it
   if (%count != 0)
   {
      commandToClient(%client, 'SetHappyModData', $HMOD_PRODUCE, %count);
   }
}

// lost happiness due to bandit attacks
function SendHappyBanditMod(%client)
{
   // loop through the happiness attack list, look for any bandit
   // attack-related datablock attached to the attack timers
   %data = 0;
   for (%index = 0; %index < $HappinessAttackCount; %index++)
   {
      %timer = $HappinessAttackList[%index];
      %type = %timer.dataBlock.getName();
      if (%type $= "BanditAttackData")
      {
         %data += %timer.happiness;
      }
   }
   
   // send the bandit attack happiness mods
   if (%data != 0)
   {
      commandToClient(%client, 'SetHappyModData', $HMOD_BANDITATTACK, %data);
   }
}

// lost happiness due to animal attacks
function SendHappyAnimalMod(%client)
{
   // not implemented
}

// if taxes are set, tell the client
function SendHappyTaxesMod(%client)
{
   // get the building list for tax rates
   %list = slgGetServerBuildingList();
   %count = %list.getSize();
   
   // look for a building that has taxes
   for (%index = 0; %index < %count; %index++)
   {
      // make sure the object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // make sure the building is in the production state
      %component = slgQueryInterface(%object, $CID_BUILDING);
      if (%component.getState() != $BuildingState::Production)
      {
         continue;
      }
      
      // check if this building has low taxes
      if (%component.taxLevel $= "lowtax")
      {
         %data = getWord(GameResourceData.taxHappiness, 0);
         commandToClient(%client, 'SetHappyModData', $HMOD_TAXES, "Low " @ %data);
      }
      // check if this building has high taxes
      else if (%component.taxLevel $= "hightax")
      {
         %data = getWord(GameResourceData.taxHappiness, 1);
         commandToClient(%client, 'SetHappyModData', $HMOD_TAXES, "High " @ %data);
      }
   }
}

// if a parade is modifying happiness, notify the client
function SendHappyParadeMod(%client)
{
   // loop through the happiness attack list, look for any parade
   // datablock attached to the attack timers
   %data = 0;
   for (%index = 0; %index < $HappinessAttackCount; %index++)
   {
      %timer = $HappinessAttackList[%index];
      %type = %timer.dataBlock.getName();
      if (%type $= "ParadeCmdData")
      {
         %data += %timer.happiness;
      }
   }
   
   // send the parade happiness mods
   if (%data != 0)
   {
      commandToClient(%client, 'SetHappyModData', $HMOD_PARADE, %data);
   }
}
